import React, { useState, useRef, useEffect } from 'react';
import Hls from 'hls.js';
import { AlertCircle, RefreshCcw } from 'lucide-react';
import VideoControls from './VideoControls';

interface VideoPlayerProps {
  src: string;
  onRefresh?: () => void;
}

const VideoPlayer: React.FC<VideoPlayerProps> = ({ src, onRefresh }) => {
  const [isPlaying, setIsPlaying] = useState(false);
  const [isMuted, setIsMuted] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [retryCount, setRetryCount] = useState(0);
  const videoRef = useRef<HTMLVideoElement>(null);
  const hlsRef = useRef<Hls | null>(null);

  const initializePlayer = () => {
    const video = videoRef.current;
    if (!video) return;

    if (hlsRef.current) {
      hlsRef.current.destroy();
      hlsRef.current = null;
    }

    setIsLoading(true);
    setError(null);

    const isHLSStream = src.includes('.m3u8');
    
    if (isHLSStream && Hls.isSupported()) {
      const hls = new Hls({
        enableWorker: true,
        lowLatencyMode: true,
        manifestLoadingTimeOut: 10000,
        manifestLoadingMaxRetry: 3,
        levelLoadingTimeOut: 10000,
        levelLoadingMaxRetry: 3
      });
      
      hls.loadSource(src);
      hls.attachMedia(video);
      
      hls.on(Hls.Events.MANIFEST_PARSED, () => {
        setIsLoading(false);
        if (isPlaying) video.play();
      });
      
      hls.on(Hls.Events.ERROR, (_, data) => {
        if (data.fatal) {
          switch (data.type) {
            case Hls.ErrorTypes.NETWORK_ERROR:
              setError('Network error while loading stream. Please check your connection.');
              break;
            case Hls.ErrorTypes.MEDIA_ERROR:
              if (retryCount < 3) {
                hls.recoverMediaError();
                setRetryCount(prev => prev + 1);
                return;
              }
              setError('Media error: stream format not supported');
              break;
            default:
              setError('Failed to load stream');
          }
          setIsLoading(false);
        }
      });
      
      hlsRef.current = hls;
    } else {
      video.src = src;
      video.load();
    }
  };

  useEffect(() => {
    initializePlayer();
    setRetryCount(0);
    return () => {
      if (hlsRef.current) {
        hlsRef.current.destroy();
        hlsRef.current = null;
      }
    };
  }, [src]);

  const handleRetry = () => {
    setRetryCount(0);
    initializePlayer();
  };

  const handleRefresh = () => {
    if (onRefresh) {
      onRefresh();
    } else {
      handleRetry();
    }
  };

  const togglePlay = () => {
    if (videoRef.current) {
      if (isPlaying) {
        videoRef.current.pause();
      } else {
        videoRef.current.play().catch(err => {
          setError('Failed to play video: ' + err.message);
        });
      }
      setIsPlaying(!isPlaying);
    }
  };

  const toggleMute = () => {
    if (videoRef.current) {
      videoRef.current.muted = !isMuted;
      setIsMuted(!isMuted);
    }
  };

  const toggleFullscreen = () => {
    if (videoRef.current) {
      if (document.fullscreenElement) {
        document.exitFullscreen();
      } else {
        videoRef.current.requestFullscreen();
      }
    }
  };

  if (error) {
    return (
      <div className="bg-red-500/10 border border-red-500/20 rounded-lg p-4 sm:p-6 text-center">
        <AlertCircle className="w-10 h-10 sm:w-12 sm:h-12 text-red-500 mx-auto mb-3 sm:mb-4" />
        <p className="text-red-500 text-sm sm:text-base">{error}</p>
        <p className="text-xs sm:text-sm text-red-400 mt-2">Please check if the stream is accessible</p>
        <button
          onClick={handleRetry}
          className="mt-4 px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition flex items-center gap-2 mx-auto text-sm sm:text-base"
        >
          <RefreshCcw className="w-4 h-4" />
          Retry Connection
        </button>
      </div>
    );
  }

  return (
    <div className="relative max-w-3xl mx-auto">
      <video
        ref={videoRef}
        className="w-full aspect-video rounded-lg shadow-lg bg-black"
        onError={() => setError('Failed to load video stream')}
        onCanPlay={() => {
          setIsLoading(false);
          setError(null);
        }}
        onPlay={() => setIsPlaying(true)}
        onPause={() => setIsPlaying(false)}
        controls={false}
        playsInline
      />
      
      {isLoading && (
        <div className="absolute inset-0 flex items-center justify-center bg-black/50 rounded-lg">
          <div className="animate-spin rounded-full h-10 w-10 sm:h-12 sm:w-12 border-4 border-white border-t-transparent"></div>
        </div>
      )}

      <VideoControls
        isPlaying={isPlaying}
        isMuted={isMuted}
        isLoading={isLoading}
        onPlayPause={togglePlay}
        onMuteToggle={toggleMute}
        onFullscreen={toggleFullscreen}
        onRefresh={handleRefresh}
      />
    </div>
  );
};

export default VideoPlayer;