import React, { useState } from 'react';
import VideoPlayer from './components/VideoPlayer';
import ChannelList from './components/ChannelList';
import Layout from './components/Layout';
import { Loader2 } from 'lucide-react';
import { parseM3UFile, type Channel } from './utils/m3uParser';

function App() {
  const [playlistUrl, setPlaylistUrl] = useState('https://iptv-org.github.io/iptv/index.m3u');
  const [channels, setChannels] = useState<Channel[]>([]);
  const [selectedChannel, setSelectedChannel] = useState<Channel>();
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string>();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(undefined);
    
    try {
      const parsedChannels = await parseM3UFile(playlistUrl);
      setChannels(parsedChannels);
      if (parsedChannels.length === 0) {
        setError('No channels found in playlist');
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to load playlist');
      setChannels([]);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <Layout>
      <form onSubmit={handleSubmit} className="mb-6 sm:mb-8">
        <div className="flex flex-col sm:flex-row gap-3">
          <input
            type="url"
            value={playlistUrl}
            onChange={(e) => setPlaylistUrl(e.target.value)}
            placeholder="Enter M3U playlist URL..."
            className="flex-1 px-4 py-2 rounded-lg bg-gray-800 text-white border border-gray-700 focus:outline-none focus:border-blue-500 text-sm sm:text-base"
            required
          />
          <button 
            type="submit"
            disabled={isLoading}
            className="px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2 text-sm sm:text-base"
          >
            {isLoading && <Loader2 className="w-4 h-4 sm:w-5 sm:h-5 animate-spin" />}
            Load Playlist
          </button>
        </div>
      </form>

      {error && (
        <div className="mb-6 sm:mb-8 p-4 bg-red-500/10 border border-red-500/20 rounded-lg text-red-400 text-sm sm:text-base">
          {error}
        </div>
      )}

      <div className="grid grid-cols-1 lg:grid-cols-[1fr,350px] gap-4 sm:gap-6 lg:gap-8">
        <div>
          {selectedChannel ? (
            <VideoPlayer 
              src={selectedChannel.url}
              onRefresh={() => {
                // Reselect the channel to refresh the stream
                const channel = selectedChannel;
                setSelectedChannel(undefined);
                setTimeout(() => setSelectedChannel(channel), 0);
              }}
            />
          ) : (
            <div className="aspect-video bg-gray-800 rounded-lg flex items-center justify-center text-gray-400 text-sm sm:text-base">
              <p>Select a channel to start watching</p>
            </div>
          )}
        </div>
        
        {channels.length > 0 && (
          <ChannelList
            channels={channels}
            onChannelSelect={setSelectedChannel}
            selectedChannel={selectedChannel}
          />
        )}
      </div>
    </Layout>
  );
}

export default App;