<?php
function parseM3UFile($url) {
    $channels = [];
    
    // Validate URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        throw new Exception('Invalid URL format');
    }

    // Initialize cURL
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        CURLOPT_HTTPHEADER => [
            'Accept: */*',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive'
        ]
    ]);
    
    $content = curl_exec($ch);
    
    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('Failed to load playlist: ' . $error);
    }
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode === 403) {
        throw new Exception('Access denied. The playlist URL is protected or requires authentication.');
    } else if ($httpCode !== 200) {
        throw new Exception('Failed to load playlist: HTTP error ' . $httpCode);
    }

    if (empty($content)) {
        throw new Exception('Empty playlist received');
    }

    // Handle potential gzip encoding
    if (strpos($content, "\x1f\x8b") === 0) {
        $content = gzdecode($content);
    }

    // Parse M3U content
    $lines = explode("\n", $content);
    $currentTitle = '';
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        if (empty($line)) continue;
        
        if (strpos($line, '#EXTINF:') === 0) {
            preg_match('/,(.+)$/', $line, $matches);
            $currentTitle = isset($matches[1]) ? $matches[1] : 'Unnamed Channel';
        } elseif (preg_match('/^https?:\/\//i', $line)) {
            $channels[] = [
                'title' => $currentTitle,
                'url' => $line
            ];
        }
    }
    
    if (empty($channels)) {
        throw new Exception('No valid channels found in the playlist');
    }
    
    return $channels;
}

function sanitizeOutput($text) {
    return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
}